﻿unit tb.AppLogger;

interface

uses
  System.SysUtils,
  System.Classes,
  System.SyncObjs,
  System.IOUtils;

procedure LogInfo(const Msg: string);
procedure LogError(const Msg: string);
procedure LogException(DebugMode: Boolean);

implementation

const
  INFO_FILE = 'InfoLog.txt';
  ERROR_FILE = 'ErrorLog.txt';
  MAX_LINES = 1000;

var
  InfoCS: TCriticalSection;
  ErrorCS: TCriticalSection;
  LOG_DIR: TFileName;

procedure EnsureLogDir;
begin
  LOG_DIR := TPath.GetDirectoryName(ParamStr(0));
  if not DirectoryExists(LOG_DIR) then
    ForceDirectories(LOG_DIR);
end;

procedure LogException(DebugMode: Boolean);
var
  E: Exception;
begin
  if not DebugMode then
    Exit;

  E := Exception(ExceptObject);
  if Assigned(E) then
    LogError('Exception: Class=' + E.ClassName + '  ->  Message=' + E.Message);
end;

procedure WriteRingLog(const FileName, Prefix, Msg: string; CS: TCriticalSection);
var
  FullPath: string;
  Lines: TStringList;
  Line: string;
begin
  EnsureLogDir;
  FullPath := IncludeTrailingPathDelimiter(LOG_DIR) + FileName;

  CS.Enter;
  try
    Lines := TStringList.Create;
    try
      if FileExists(FullPath) then
        Lines.LoadFromFile(FullPath);

      // Neue Zeile erzeugen
      Line := Format('%s [%s] %s',
        [Prefix, FormatDateTime('yyyy-mm-dd hh:nn:ss.zzz', Now), Msg]);

      // Neue Zeile OBEN einfügen
      Lines.Insert(0, Line);

      // Wenn mehr als MAX_LINES → letzte Zeile löschen
      while Lines.Count > MAX_LINES do
        Lines.Delete(Lines.Count - 1);

      // Datei speichern
      Lines.SaveToFile(FullPath);
    finally
      Lines.Free;
    end;
  finally
    CS.Leave;
  end;
end;

procedure LogInfo(const Msg: string);
begin
  try
    WriteRingLog(INFO_FILE, 'INFO', Msg, InfoCS);
  except
    // Logger wirft nie Exceptions
  end;
end;

procedure LogError(const Msg: string);
begin
  try
    WriteRingLog(ERROR_FILE, 'ERROR', Msg, ErrorCS);
  except
  end;
end;

initialization
  InfoCS := TCriticalSection.Create;
  ErrorCS := TCriticalSection.Create;

finalization
  InfoCS.Free;
  ErrorCS.Free;

end.

